package gov.va.vinci.dart.rule;

import gov.va.vinci.dart.biz.DartRequest;
import gov.va.vinci.dart.biz.PreparatoryRequest;
import gov.va.vinci.dart.biz.Request;
import gov.va.vinci.dart.biz.DataSource;
import gov.va.vinci.dart.biz.DocumentTemplate;
import gov.va.vinci.dart.biz.Location;
import gov.va.vinci.dart.biz.Participant;

/**
 * Class CDWDocumentRuleEvaluator.
 *
 */
public class CDWDocumentRuleEvaluator extends AbstractDocumentRuleEvaluator {

    /**
     * Is a specific document required for a site?.
     *
     */
    @Override
    public boolean isRequired(Request incomingRequest, DataSource dataSource, DocumentTemplate documentTemplate,
            Location location, final boolean isIndependentOnly) {

        if (incomingRequest == null) {
            return false;
        }

        Location primarySite = incomingRequest.getPrimaryLocation();

        if (isDartRequest(incomingRequest)) {
            DartRequest dartRequest = (DartRequest) incomingRequest;

            // IRB Approval of Waiver of HIPAA is required of all sites if the HIPAA waiver is true
            if (documentTemplate.getName().startsWith("IRB Approval of Waiver of HIPAA") 
                    && dartRequest.isHipaaWaiver()) {
                return true;
            }

            // Research and Development (RD) Committee Approval is required of all sites
            if (documentTemplate.getName().startsWith("Research and Development (RD) Committee Approval")) {
                return true;
            }

            // Research Study Institutional Review Board (IRB) Approval is required of all sites
            if (documentTemplate.getName().startsWith("Research Study Institutional Review Board (IRB) Approval")) {
                return true;
            }

            if (primarySite != null) {

                // Research Request Memo is required for the primary site
                if (documentTemplate.getName().startsWith("Research Request Memo") && location.getId() == primarySite.getId()) {
                    return true;
                }

                // Sample Informed Consent is required for the primary site if the HIPAA patient consent is true
                if (documentTemplate.getName().startsWith("Sample Informed Consent") && location.getId() == primarySite.getId()
                        && dartRequest.isHipaaConsent()) {
                    return true;
                }

                // As per DUA document item 1.10, include DUA if transferredexternal = true
                // Data Use Agreement is required for the primary site
                if (documentTemplate.getName().startsWith("Data Use Agreement") && location.getId() == primarySite.getId()
                        && dartRequest.isTransferredExternal()) {
                    return true;
                }

                // Research Protocol is required for the primary site
                if (documentTemplate.getName().startsWith("Research Protocol") && location.getId() == primarySite.getId()) {
                    return true;
                }

                // CDW-NDS Authorization Form is required for the primary site
                if (documentTemplate.getName().startsWith("CDW-Domain Checklist") && location.getId() == primarySite.getId()) {
                    return true;
                }

                // As of 8/13/2014 (Real SSN BRD item 1.2):
                // Real SSN Access Request is required for the primary site, if realSSN is selected
                if (documentTemplate.getName().startsWith("Real SSN Access Request") && location.getId() == primarySite.getId()
                        && dartRequest.isRealSSN()) {

                    // As of 2/25/2015: Real SSN Access Request no longer required for any Independent Approver Workflow
                    // if( dartRequest.getWorkflowTypeId() == WorkflowResolver.WF_NDS ) {
                    if (!isIndependentOnly) {
                        return true;
                    }
                }
            }
        } else if (isPreparatoryRequest(incomingRequest)) {
           
            if (primarySite != null) {
                if (documentTemplate.getName().startsWith(PREPARATORY_TO_RESEARCH_MEMORANDUM) 
                        && location.getId() == primarySite.getId()) {
                    return true;
                }
            }
        }

        return false;
    }

    /**
     * Is a specific document required for a participant?
     * 
     */
    @Override
    public boolean isRequired(final Request req, final DataSource ds, final DocumentTemplate temp,
            final Participant participant, final boolean isIndependentOnly) {

        // As of 3/23/2015, VA Form 9957 is no longer required for any NDS data source
        // // VA Form 9957 is required of Principal Investigator of Primary Location
        // if (participant.getPrincipalInvestigator().booleanValue() == true && req.getPrimaryLocation().getId() ==
        // participant.getLocation().getId()) {
        // if (temp.getName().startsWith("VA Form 9957") ) {
        // return true;
        // }
        // }

        return false; // end method
    }
}
